<?php
/* iCal formatting utilities */

function prepareICalText($text) {
    // Replace smart punctuation and common Unicode troublemakers
    $replacements = [
        "\xE2\x80\x98" => "'", // ‘
        "\xE2\x80\x99" => "'", // ’
        "\xE2\x80\x9C" => '"', // “
        "\xE2\x80\x9D" => '"', // ”
        "\xE2\x80\x93" => '-', // –
        "\xE2\x80\x94" => '-', // —
        "\xC2\xA0"     => ' ', // non-breaking space
        "\xC2\xAB"     => '"', // «
        "\xC2\xBB"     => '"', // »
    ];
    $text = str_replace(array_keys($replacements), array_values($replacements), $text);

    // Strip HTML and decode any entities
    $text = strip_tags($text);
    $text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');

    // Normalize to UTF-8 (with fail-safe fallback)
    $text = mb_convert_encoding($text, 'UTF-8', 'UTF-8');

    // Replace real newlines with literal "\n"
    $text = preg_replace('/\r\n|\r|\n/', '\\n', $text);

    // Collapse multiple whitespace characters
    $text = preg_replace('/\s+/', ' ', $text);

    // Escape iCalendar special characters
    $text = str_replace(['\\', ',', ';', '"'], ['\\\\', '\,', '\;', '\''], $text);

    // Fold to 75 bytes per line, as per RFC 5545
    return foldICalLine($text);
}

function foldICalLine($text, $maxLength = 75) {
    $lines = [];
    $current = '';
    $bytes = 0;

    foreach (preg_split('//u', $text, -1, PREG_SPLIT_NO_EMPTY) as $char) {
        $charBytes = strlen(mb_convert_encoding($char, 'UTF-8'));
        if ($bytes + $charBytes > $maxLength) {
            $lines[] = $current;
            $current = ' ' . $char; // Leading space for folded line
            $bytes = 1 + $charBytes;
        } else {
            $current .= $char;
            $bytes += $charBytes;
        }
    }
    $lines[] = $current;

    return implode("\r\n", $lines);
}
?>